<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Http\Requests\CustomerRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use ZipArchive;

class CustomerController extends Controller
{
    public function index(Request $request)
    {
        // Start query
        $query = Customer::withCount(['installments', 'payments'])
            ->withSum('payments', 'amount');

        // Apply filters
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('documents')) {
            if ($request->documents == 'complete') {
                $query->withCompleteDocuments();
            } elseif ($request->documents == 'incomplete') {
                $query->withIncompleteDocuments();
            }
        }

        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->from_date);
        }

        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->to_date);
        }

        if ($request->filled('search')) {
            $query->search($request->search);
        }

        $customers = $query->orderBy('created_at', 'desc')->paginate(10);

        // Add additional stats for dashboard
        $totalCustomers = Customer::count();
        $activeCustomers = Customer::active()->count();
        $customersWithMissingDocs = Customer::withIncompleteDocuments()->count();
        $thisMonthCustomers = Customer::where('created_at', '>=', now()->startOfMonth())->count();

        return view('customers.index', compact(
            'customers',
            'totalCustomers',
            'activeCustomers',
            'customersWithMissingDocs',
            'thisMonthCustomers'
        ));
    }

    public function create()
    {
        return view('customers.create');
    }

    public function store(CustomerRequest $request)
    {
        try {
            DB::beginTransaction();

            // Create customer
            $customer = new Customer();
            $customer->fill($request->except([
                'cnic_front', 'cnic_back',
                'customer_picture', 'other_documents'
            ]));

            // Upload documents
            $uploads = $this->uploadDocuments($request);
            $customer->fill($uploads);

            $customer->save();

            DB::commit();

            if ($request->has('add_another')) {
                return redirect()->route('customers.create')
                    ->with('success', 'Customer created successfully. You can add another.');
            }

            return redirect()->route('customers.show', $customer->id)
                ->with('success', 'Customer created successfully with all documents.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', 'Error creating customer: ' . $e->getMessage());
        }
    }

    public function show(Customer $customer)
    {
        // Load relationships with proper counts and sums
        $customer->loadCount(['installments', 'payments']);
        $customer->loadSum('payments', 'amount');

        // Calculate additional stats
        $customer->total_paid = $customer->payments()->where('status', 'completed')->sum('amount');
        $customer->total_pending = $customer->installments()->where('status', '!=', 'paid')->sum('amount');

        // Get properties with progress
        $properties = $customer->properties()->withCount(['installments'])->get();

        // Load recent installments and payments
        $customer->load(['installments.property', 'payments.installment.property']);

        return view('customers.show', compact('customer', 'properties'));
    }

    public function edit(Customer $customer)
    {
        return view('customers.edit', compact('customer'));
    }

    public function update(CustomerRequest $request, Customer $customer)
    {
        try {
            DB::beginTransaction();

            $oldDocuments = [
                'cnic_front' => $customer->cnic_front,
                'cnic_back' => $customer->cnic_back,
                'customer_picture' => $customer->customer_picture,
                'other_documents' => $customer->other_documents ?? [],
            ];

            // Update basic info
            $customer->fill($request->except([
                'cnic_front', 'cnic_back',
                'customer_picture', 'other_documents'
            ]));

            // Handle document updates
            $uploads = $this->uploadDocuments($request);

            // Merge new uploads with existing documents
            foreach ($uploads as $field => $value) {
                if ($field === 'other_documents' && !empty($value)) {
                    // Merge old and new documents
                    $existingDocs = $oldDocuments['other_documents'] ?? [];
                    $newDocs = is_array($value) ? $value : [$value];
                    $customer->$field = array_merge($existingDocs, $newDocs);
                } elseif (!empty($value)) {
                    // Delete old file if exists
                    if (!empty($oldDocuments[$field])) {
                        Storage::disk('public')->delete($oldDocuments[$field]);
                    }
                    $customer->$field = $value;
                }
            }

            $customer->save();

            DB::commit();

            return redirect()->route('customers.show', $customer->id)
                ->with('success', 'Customer updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', 'Error updating customer: ' . $e->getMessage());
        }
    }

    public function destroy(Customer $customer)
    {
        try {
            DB::beginTransaction();

            // Delete all associated documents
            $customer->deleteAllDocuments();

            // Delete the customer
            $customer->delete();

            DB::commit();

            return redirect()->route('customers.index')
                ->with('success', 'Customer and all associated documents deleted successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error deleting customer: ' . $e->getMessage());
        }
    }

    /**
     * Delete specific document
     */
    public function deleteDocument(Request $request, Customer $customer)
    {
        $request->validate([
            'document_type' => 'required|in:cnic_front,cnic_back,customer_picture,other_documents',
            'document_index' => 'nullable|integer',
        ]);

        $type = $request->document_type;
        $index = $request->document_index;

        try {
            DB::beginTransaction();

            $result = $customer->deleteDocument($type, $index);

            DB::commit();

            if ($result) {
                return back()->with('success', 'Document deleted successfully.');
            }

            return back()->with('warning', 'Document not found.');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error deleting document: ' . $e->getMessage());
        }
    }

    /**
     * Download specific document
     */
    public function downloadDocument(Customer $customer, $type, $index = null)
    {
        try {
            $downloadData = $customer->getDownloadData($type, $index);

            if ($downloadData && file_exists($downloadData['path'])) {
                return response()->download(
                    $downloadData['path'],
                    $downloadData['filename']
                );
            }

            return back()->with('error', 'Document not found or has been deleted.');

        } catch (\Exception $e) {
            return back()->with('error', 'Error downloading document: ' . $e->getMessage());
        }
    }

    /**
     * Download all documents as zip
     */
    public function downloadAllDocuments(Customer $customer)
    {
        try {
            $documents = $customer->getAllDocumentsData();

            if (empty($documents)) {
                return back()->with('error', 'No documents found for this customer.');
            }

            // Create zip file
            $zipFileName = 'customer_' . $customer->id . '_documents_' . date('Ymd_His') . '.zip';
            $zipPath = storage_path('app/temp/' . $zipFileName);

            // Create temp directory if not exists
            if (!file_exists(dirname($zipPath))) {
                mkdir(dirname($zipPath), 0755, true);
            }

            // Create zip archive
            $zip = new ZipArchive();
            if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {
                foreach ($documents as $document) {
                    if (file_exists($document['path'])) {
                        $zip->addFile($document['path'], $document['filename']);
                    }
                }
                $zip->close();

                // Download zip file
                return response()->download($zipPath)->deleteFileAfterSend(true);
            }

            return back()->with('error', 'Failed to create zip archive.');

        } catch (\Exception $e) {
            return back()->with('error', 'Error creating zip file: ' . $e->getMessage());
        }
    }

    /**
     * Export customers to CSV
     */
    public function export(Request $request)
    {
        try {
            $query = Customer::withCount(['installments', 'payments'])
                ->withSum('payments', 'amount');

            // Apply filters if any
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }

            $customers = $query->get();

            $fileName = 'customers_export_' . date('Ymd_His') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
            ];

            $callback = function() use ($customers) {
                $file = fopen('php://output', 'w');

                // Add CSV headers
                fputcsv($file, [
                    'ID', 'Name', 'Email', 'Phone', 'CNIC', 'Status',
                    'Total Installments', 'Total Paid', 'Document Status',
                    'Created Date', 'Address'
                ]);

                // Add data rows
                foreach ($customers as $customer) {
                    fputcsv($file, [
                        $customer->id,
                        $customer->name,
                        $customer->email,
                        $customer->phone,
                        $customer->cnic,
                        $customer->status,
                        $customer->installments_count,
                        $customer->payments_sum_amount ?? 0,
                        $customer->has_complete_documents ? 'Complete' : 'Incomplete',
                        $customer->created_at->format('Y-m-d'),
                        $customer->address
                    ]);
                }

                fclose($file);
            };

            return response()->stream($callback, 200, $headers);

        } catch (\Exception $e) {
            return back()->with('error', 'Error exporting data: ' . $e->getMessage());
        }
    }

    /**
     * Handle multiple document uploads
     */
    private function uploadDocuments($request)
    {
        $uploads = [];

        // Upload CNIC Front
        if ($request->hasFile('cnic_front')) {
            $uploads['cnic_front'] = $request->file('cnic_front')
                ->store(Customer::CNIC_FRONT_PATH, 'public');
        }

        // Upload CNIC Back
        if ($request->hasFile('cnic_back')) {
            $uploads['cnic_back'] = $request->file('cnic_back')
                ->store(Customer::CNIC_BACK_PATH, 'public');
        }

        // Upload Customer Picture
        if ($request->hasFile('customer_picture')) {
            $uploads['customer_picture'] = $request->file('customer_picture')
                ->store(Customer::CUSTOMER_PICTURE_PATH, 'public');
        }

        // Upload Other Documents (multiple files)
        if ($request->hasFile('other_documents')) {
            $otherDocs = [];
            foreach ($request->file('other_documents') as $document) {
                $otherDocs[] = $document->store(Customer::OTHER_DOCUMENTS_PATH, 'public');
            }
            if (!empty($otherDocs)) {
                $uploads['other_documents'] = $otherDocs;
            }
        }

        return $uploads;
    }
}
