<?php

namespace App\Http\Controllers;

use App\Models\Installment;
use App\Models\Payment;
use Illuminate\Http\Request;

class InstallmentController extends Controller
{
    public function index(Request $request)
    {
        $query = Installment::with(['property', 'customer']);

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('property_id')) {
            $query->where('property_id', $request->property_id);
        }

        $installments = $query->orderBy('due_date', 'asc')
            ->paginate(20);

        return view('installments.index', compact('installments'));
    }

    public function show(Installment $installment)
    {
        $installment->load(['property', 'customer', 'payments']);

        return view('installments.show', compact('installment'));
    }

    public function edit(Installment $installment)
    {
        return view('installments.edit', compact('installment'));
    }

    public function update(Request $request, Installment $installment)
    {
        $validated = $request->validate([
            'due_date' => 'required|date',
            'amount' => 'required|numeric|min:0',
            'status' => 'required|in:pending,paid,overdue,partial',
            'notes' => 'nullable|string',
        ]);

        $installment->update($validated);

        return redirect()->route('installments.show', $installment)
            ->with('success', 'Installment updated successfully.');
    }

    public function markAsPaid(Installment $installment)
    {
        $installment->update([
            'paid_amount' => $installment->amount,
            'status' => 'paid',
        ]);

        return back()->with('success', 'Installment marked as paid.');
    }
}
