<?php

namespace App\Http\Controllers;

use App\Models\Property;
use App\Models\Customer;
use App\Models\Installment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PropertyController extends Controller
{
    public function index()
    {
        $properties = Property::withCount('installments')
            ->withSum('installments', 'paid_amount')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('properties.index', compact('properties'));
    }

    public function create()
    {
        $customers = Customer::where('status', 'active')->get();
        return view('properties.create', compact('customers'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'location' => 'required|string|max:500',
            'total_price' => 'required|numeric|min:0',
            'down_payment' => 'required|numeric|min:0',
            'installment_years' => 'required|integer|min:1|max:30',
            'description' => 'nullable|string',
            'customer_id' => 'required|exists:customers,id',
        ]);

        DB::beginTransaction();
        try {
            $installmentAmount = ($validated['total_price'] - $validated['down_payment']) /
                                ($validated['installment_years'] * 12);

            $property = Property::create([
                'name' => $validated['name'],
                'location' => $validated['location'],
                'total_price' => $validated['total_price'],
                'down_payment' => $validated['down_payment'],
                'installment_years' => $validated['installment_years'],
                'installment_amount' => round($installmentAmount, 2),
                'total_installments' => $validated['installment_years'] * 12,
                'description' => $validated['description'],
                'status' => 'active',
            ]);

            // Create installments
            $startDate = now();
            for ($i = 1; $i <= $property->total_installments; $i++) {
                Installment::create([
                    'property_id' => $property->id,
                    'customer_id' => $validated['customer_id'],
                    'start_date' => $startDate,
                    'end_date' => $startDate->copy()->addMonth(),
                    'installment_number' => $i,
                    'due_date' => $startDate->copy()->addMonth(),
                    'amount' => $property->installment_amount,
                    'status' => 'pending',
                ]);
                $startDate->addMonth();
            }

            DB::commit();

            return redirect()->route('properties.index')
                ->with('success', 'Property created successfully with installments.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to create property: ' . $e->getMessage());
        }
    }

    public function show(Property $property)
    {
        $property->load(['installments.customer', 'installments.payments']);
        $installments = $property->installments()->paginate(10);

        return view('properties.show', compact('property', 'installments'));
    }

    public function edit(Property $property)
    {
        return view('properties.edit', compact('property'));
    }

    public function update(Request $request, Property $property)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'location' => 'required|string|max:500',
            'status' => 'required|in:active,completed,cancelled',
            'description' => 'nullable|string',
        ]);

        $property->update($validated);

        return redirect()->route('properties.index')
            ->with('success', 'Property updated successfully.');
    }

    public function destroy(Property $property)
    {
        $property->delete();

        return redirect()->route('properties.index')
            ->with('success', 'Property deleted successfully.');
    }
}
