<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class CustomerRequest extends FormRequest
{
    public function authorize()
    {
        return true;
    }

    public function rules()
    {
        $customerId = $this->route('customer') ? $this->route('customer')->id : null;

        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:customers,email,' . $customerId,
            'phone' => 'required|string|max:20|unique:customers,phone,' . $customerId,
            'cnic' => 'required|string|unique:customers,cnic,' . $customerId,
            'address' => 'required|string',
            'date_of_birth' => 'nullable|date',
            'status' => 'required|in:active,inactive,blacklisted',

            // Document validation rules
            'cnic_front' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'cnic_back' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'customer_picture' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'other_documents.*' => 'nullable|file|mimes:pdf,jpeg,png,jpg,doc,docx|max:5120',
        ];

        return $rules;
    }

    public function messages()
    {
        return [
            'name.required' => 'Customer name is required',
            'email.email' => 'Please enter a valid email address',
            'email.unique' => 'This email is already registered',
            'phone.required' => 'Phone number is required',
            'phone.unique' => 'This phone number is already registered',
            'cnic.required' => 'CNIC number is required',
            'cnic.unique' => 'This CNIC is already registered',
            'address.required' => 'Address is required',
            'status.required' => 'Status is required',

            'cnic_front.max' => 'CNIC front image must not exceed 2MB',
            'cnic_back.max' => 'CNIC back image must not exceed 2MB',
            'customer_picture.max' => 'Customer picture must not exceed 2MB',
            'other_documents.*.max' => 'Each document must not exceed 5MB',
            'other_documents.*.mimes' => 'Allowed document formats: PDF, JPEG, PNG, JPG, DOC, DOCX',
        ];
    }

    public function attributes()
    {
        return [
            'name' => 'Full Name',
            'email' => 'Email Address',
            'phone' => 'Phone Number',
            'cnic' => 'CNIC Number',
            'address' => 'Complete Address',
            'date_of_birth' => 'Date of Birth',
            'status' => 'Status',
            'cnic_front' => 'CNIC Front Side',
            'cnic_back' => 'CNIC Back Side',
            'customer_picture' => 'Customer Picture',
            'other_documents' => 'Other Documents',
        ];
    }
}
