<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\Storage;

class Customer extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'phone',
        'cnic',
        'address',
        'date_of_birth',
        'status',
        'cnic_front',
        'cnic_back',
        'customer_picture',
        'other_documents',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'date_of_birth' => 'date',
        'other_documents' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Storage path constants for organized file storage
     */
    const CNIC_FRONT_PATH = 'customers/cnic/front';
    const CNIC_BACK_PATH = 'customers/cnic/back';
    const CUSTOMER_PICTURE_PATH = 'customers/pictures';
    const OTHER_DOCUMENTS_PATH = 'customers/documents';

    /**
     * Maximum file sizes in KB
     */
    const MAX_CNIC_SIZE = 2048; // 2MB
    const MAX_PICTURE_SIZE = 2048; // 2MB
    const MAX_DOCUMENT_SIZE = 5120; // 5MB

    /**
     * Allowed file extensions
     */
    const ALLOWED_IMAGE_EXTENSIONS = ['jpg', 'jpeg', 'png', 'gif'];
    const ALLOWED_DOCUMENT_EXTENSIONS = ['pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png'];

    /**
     * Relationships
     */
    public function properties()
    {
        return $this->belongsToMany(Property::class, 'customer_property')
            ->withPivot('purchase_date', 'total_amount', 'down_payment', 'installment_amount','total_installments',
            'payment_frequency',
            'notes')
            ->withTimestamps();
    }

    public function installments()
    {
        return $this->hasMany(Installment::class);
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Scopes - CORRECTED VERSION
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeInactive($query)
    {
        return $query->where('status', 'inactive');
    }

    public function scopeBlacklisted($query)
    {
        return $query->where('status', 'blacklisted');
    }

    // Customers with ALL required documents
    public function scopeWithCompleteDocuments($query)
    {
        return $query->whereNotNull('cnic_front')
            ->whereNotNull('cnic_back')
            ->whereNotNull('customer_picture');
    }

    // Customers missing ANY required document
    public function scopeWithIncompleteDocuments($query)
    {
        return $query->where(function($q) {
            $q->whereNull('cnic_front')
              ->orWhereNull('cnic_back')
              ->orWhereNull('customer_picture');
        });
    }

    // Customers missing specific document
    public function scopeMissingCnicFront($query)
    {
        return $query->whereNull('cnic_front');
    }

    public function scopeMissingCnicBack($query)
    {
        return $query->whereNull('cnic_back');
    }

    public function scopeMissingPicture($query)
    {
        return $query->whereNull('customer_picture');
    }

    // Search by name, email, phone, or CNIC
    public function scopeSearch($query, $searchTerm)
    {
        return $query->where(function($q) use ($searchTerm) {
            $q->where('name', 'LIKE', "%{$searchTerm}%")
              ->orWhere('email', 'LIKE', "%{$searchTerm}%")
              ->orWhere('phone', 'LIKE', "%{$searchTerm}%")
              ->orWhere('cnic', 'LIKE', "%{$searchTerm}%");
        });
    }

    /**
     * Accessors
     */
    protected function name(): Attribute
    {
        return Attribute::make(
            get: fn (string $value) => ucwords(strtolower($value)),
        );
    }

    protected function cnic(): Attribute
    {
        return Attribute::make(
            get: fn (string $value) => preg_replace('/(\d{5})(\d{7})(\d{1})/', '$1-$2-$3', $value),
            set: fn (string $value) => str_replace('-', '', $value),
        );
    }

    protected function phone(): Attribute
    {
        return Attribute::make(
            get: fn (string $value) => '+92 ' . substr($value, 0, 3) . ' ' . substr($value, 3, 7),
            set: fn (string $value) => preg_replace('/[^0-9]/', '', $value),
        );
    }

    protected function cnicFrontUrl(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->cnic_front ? Storage::url($this->cnic_front) : null,
        );
    }

    protected function cnicBackUrl(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->cnic_back ? Storage::url($this->cnic_back) : null,
        );
    }

    protected function customerPictureUrl(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->customer_picture ? Storage::url($this->customer_picture) : null,
        );
    }

    protected function otherDocumentsUrls(): Attribute
    {
        return Attribute::make(
            get: function () {
                if (empty($this->other_documents)) {
                    return [];
                }

                return array_map(function ($document) {
                    return [
                        'path' => $document,
                        'url' => Storage::url($document),
                        'name' => basename($document),
                        'extension' => pathinfo($document, PATHINFO_EXTENSION),
                        'icon' => $this->getFileIcon($document),
                    ];
                }, $this->other_documents);
            }
        );
    }

    /**
     * Get file icon based on extension
     */
    private function getFileIcon($filePath)
    {
        $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));

        $icons = [
            'pdf' => 'fa-file-pdf text-danger',
            'doc' => 'fa-file-word text-primary',
            'docx' => 'fa-file-word text-primary',
            'jpg' => 'fa-file-image text-success',
            'jpeg' => 'fa-file-image text-success',
            'png' => 'fa-file-image text-success',
            'gif' => 'fa-file-image text-success',
        ];

        return $icons[$extension] ?? 'fa-file text-secondary';
    }

    /**
     * Check if customer has all required documents
     */
    public function getHasCompleteDocumentsAttribute()
    {
        return !empty($this->cnic_front) &&
               !empty($this->cnic_back) &&
               !empty($this->customer_picture);
    }

    /**
     * Get document completion percentage
     */
    public function getDocumentCompletionAttribute()
    {
        $documents = [
            'cnic_front' => !empty($this->cnic_front),
            'cnic_back' => !empty($this->cnic_back),
            'customer_picture' => !empty($this->customer_picture),
        ];

        $completed = count(array_filter($documents));
        $total = count($documents);

        return round(($completed / $total) * 100, 2);
    }

    /**
     * Get total number of uploaded documents
     */
    public function getTotalDocumentsAttribute()
    {
        $basicDocs = ($this->cnic_front ? 1 : 0) +
                    ($this->cnic_back ? 1 : 0) +
                    ($this->customer_picture ? 1 : 0);

        $otherDocs = !empty($this->other_documents) ? count($this->other_documents) : 0;

        return $basicDocs + $otherDocs;
    }

    /**
     * Handle multiple document uploads
     */
    public function uploadDocuments($request)
    {
        $uploads = [];

        // Upload CNIC Front
        if ($request->hasFile('cnic_front')) {
            $uploads['cnic_front'] = $request->file('cnic_front')
                ->store(self::CNIC_FRONT_PATH, 'public');
        }

        // Upload CNIC Back
        if ($request->hasFile('cnic_back')) {
            $uploads['cnic_back'] = $request->file('cnic_back')
                ->store(self::CNIC_BACK_PATH, 'public');
        }

        // Upload Customer Picture
        if ($request->hasFile('customer_picture')) {
            $uploads['customer_picture'] = $request->file('customer_picture')
                ->store(self::CUSTOMER_PICTURE_PATH, 'public');
        }

        // Upload Other Documents (multiple files)
        if ($request->hasFile('other_documents')) {
            $otherDocs = [];
            foreach ($request->file('other_documents') as $document) {
                $otherDocs[] = $document->store(self::OTHER_DOCUMENTS_PATH, 'public');
            }
            if (!empty($otherDocs)) {
                $uploads['other_documents'] = $otherDocs;
            }
        }

        return $uploads;
    }

    /**
     * Delete specific document
     */
    public function deleteDocument($type, $index = null)
    {
        try {
            if ($type === 'other_documents' && $index !== null) {
                // Delete specific document from array
                $documents = $this->other_documents ?? [];
                if (isset($documents[$index])) {
                    Storage::disk('public')->delete($documents[$index]);
                    unset($documents[$index]);
                    $this->other_documents = array_values($documents);
                    $this->save();
                    return true;
                }
            } elseif (in_array($type, ['cnic_front', 'cnic_back', 'customer_picture'])) {
                // Delete single document
                if ($this->$type) {
                    Storage::disk('public')->delete($this->$type);
                    $this->$type = null;
                    $this->save();
                    return true;
                }
            }

            return false;

        } catch (\Exception $e) {
            throw new \Exception("Failed to delete document: " . $e->getMessage());
        }
    }

    /**
     * Delete all documents associated with this customer
     */
    public function deleteAllDocuments()
    {
        try {
            // Delete single file documents
            $documentsToDelete = [
                'cnic_front' => $this->cnic_front,
                'cnic_back' => $this->cnic_back,
                'customer_picture' => $this->customer_picture,
            ];

            foreach ($documentsToDelete as $field => $path) {
                if (!empty($path) && Storage::disk('public')->exists($path)) {
                    Storage::disk('public')->delete($path);
                }
            }

            // Delete other documents array
            if (!empty($this->other_documents)) {
                foreach ($this->other_documents as $documentPath) {
                    if (Storage::disk('public')->exists($documentPath)) {
                        Storage::disk('public')->delete($documentPath);
                    }
                }
            }

            return true;

        } catch (\Exception $e) {
            throw new \Exception("Failed to delete documents: " . $e->getMessage());
        }
    }

    /**
     * Get download data for a specific document
     */
    public function getDownloadData($type, $index = null)
    {
        if ($type === 'other_documents' && $index !== null) {
            $documents = $this->other_documents ?? [];
            if (isset($documents[$index]) && Storage::disk('public')->exists($documents[$index])) {
                $path = storage_path('app/public/' . $documents[$index]);
                $filename = $this->getDownloadFilename($type, $index, $path);
                return [
                    'path' => $path,
                    'filename' => $filename,
                ];
            }
        } elseif (in_array($type, ['cnic_front', 'cnic_back', 'customer_picture'])) {
            if ($this->$type && Storage::disk('public')->exists($this->$type)) {
                $path = storage_path('app/public/' . $this->$type);
                $filename = $this->getDownloadFilename($type, null, $path);
                return [
                    'path' => $path,
                    'filename' => $filename,
                ];
            }
        }

        return null;
    }

    /**
     * Generate download filename
     */
    private function getDownloadFilename($type, $index = null, $path = null)
    {
        $customerName = preg_replace('/[^a-z0-9]/', '_', strtolower($this->name));
        $timestamp = date('Ymd_His');

        $typeNames = [
            'cnic_front' => 'cnic_front',
            'cnic_back' => 'cnic_back',
            'customer_picture' => 'picture',
            'other_documents' => 'document_' . ($index + 1),
        ];

        $typeName = $typeNames[$type] ?? $type;

        if ($path) {
            $extension = pathinfo($path, PATHINFO_EXTENSION);
            return "{$customerName}_{$typeName}_{$timestamp}.{$extension}";
        }

        return "{$customerName}_{$typeName}_{$timestamp}";
    }

    /**
     * Get all documents data for zip download
     */
    public function getAllDocumentsData()
    {
        $documents = [];

        // Add CNIC front
        if ($this->cnic_front) {
            $documents[] = [
                'path' => storage_path('app/public/' . $this->cnic_front),
                'filename' => $this->getDownloadFilename('cnic_front', null, $this->cnic_front),
                'type' => 'cnic_front',
            ];
        }

        // Add CNIC back
        if ($this->cnic_back) {
            $documents[] = [
                'path' => storage_path('app/public/' . $this->cnic_back),
                'filename' => $this->getDownloadFilename('cnic_back', null, $this->cnic_back),
                'type' => 'cnic_back',
            ];
        }

        // Add customer picture
        if ($this->customer_picture) {
            $documents[] = [
                'path' => storage_path('app/public/' . $this->customer_picture),
                'filename' => $this->getDownloadFilename('customer_picture', null, $this->customer_picture),
                'type' => 'customer_picture',
            ];
        }

        // Add other documents
        if (!empty($this->other_documents)) {
            foreach ($this->other_documents as $index => $document) {
                $documents[] = [
                    'path' => storage_path('app/public/' . $document),
                    'filename' => $this->getDownloadFilename('other_documents', $index, $document),
                    'type' => 'other_document',
                ];
            }
        }

        return $documents;
    }

    /**
     * Boot method for model events
     */
    protected static function boot()
    {
        parent::boot();

        // Delete all documents when customer is deleted
        static::deleting(function ($customer) {
            $customer->deleteAllDocuments();

            // Also delete related records if needed
            $customer->installments()->delete();
            $customer->payments()->delete();
            $customer->properties()->detach();
        });
    }

    /**
     * Additional business logic methods
     */
    public function getTotalPaidAmount()
    {
        return $this->payments()->where('status', 'completed')->sum('amount');
    }

    public function getTotalPendingAmount()
    {
        return $this->installments()->where('status', '!=', 'paid')->sum('amount');
    }

    public function getPaymentProgressPercentage()
    {
        $totalPaid = $this->getTotalPaidAmount();
        $totalPending = $this->getTotalPendingAmount();
        $totalAmount = $totalPaid + $totalPending;

        return $totalAmount > 0 ? round(($totalPaid / $totalAmount) * 100, 2) : 0;
    }

    /**
     * Check if email notification can be sent
     */
    public function canReceiveEmailNotifications()
    {
        return !empty($this->email) && $this->status === 'active';
    }

    /**
     * Check if SMS notification can be sent
     */
    public function canReceiveSmsNotifications()
    {
        return !empty($this->phone) && $this->status === 'active';
    }

    /**
     * Get customer's age from date of birth
     */
    public function getAgeAttribute()
    {
        if (!$this->date_of_birth) {
            return null;
        }

        return $this->date_of_birth->age;
    }

    /**
     * Get customer's initials for avatar
     */
    public function getInitialsAttribute()
    {
        $names = explode(' ', $this->name);
        $initials = '';

        foreach ($names as $name) {
            $initials .= strtoupper(substr($name, 0, 1));
        }

        return substr($initials, 0, 2);
    }


}
