@extends('layouts.app')

@section('title', 'Edit Customer: ' . $customer->name)
@section('page-title', 'Edit Customer')

@section('content')
<div class="row justify-content-center">
    <div class="col-md-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Edit Customer Information</h5>
            </div>
            <div class="card-body">
                <form method="POST" action="{{ route('customers.update', $customer->id) }}" id="customerForm" enctype="multipart/form-data">
                    @csrf
                    @method('PUT')

                    <div class="row">
                        <!-- Basic Information -->
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="name" class="form-label">Full Name *</label>
                                <input type="text" class="form-control @error('name') is-invalid @enderror"
                                       id="name" name="name" value="{{ old('name', $customer->name) }}"
                                       placeholder="Enter customer full name" required>
                                @error('name')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="email" class="form-label">Email Address</label>
                                <input type="email" class="form-control @error('email') is-invalid @enderror"
                                       id="email" name="email" value="{{ old('email', $customer->email) }}"
                                       placeholder="Enter email address">
                                @error('email')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="phone" class="form-label">Phone Number *</label>
                                <div class="input-group">
                                    <span class="input-group-text">+92</span>
                                    <input type="text" class="form-control @error('phone') is-invalid @enderror"
                                           id="phone" name="phone" value="{{ old('phone', $customer->phone) }}"
                                           placeholder="3001234567" required>
                                </div>
                                @error('phone')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="cnic" class="form-label">CNIC Number *</label>
                                <input type="text" class="form-control @error('cnic') is-invalid @enderror"
                                       id="cnic" name="cnic" value="{{ old('cnic', $customer->cnic) }}"
                                       placeholder="12345-1234567-1" required>
                                @error('cnic')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="date_of_birth" class="form-label">Date of Birth</label>
                                <input type="date" class="form-control @error('date_of_birth') is-invalid @enderror"
                                       id="date_of_birth" name="date_of_birth" value="{{ old('date_of_birth', $customer->date_of_birth ? $customer->date_of_birth->format('Y-m-d') : '') }}">
                                @error('date_of_birth')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="status" class="form-label">Status *</label>
                                <select class="form-control @error('status') is-invalid @enderror"
                                        id="status" name="status" required>
                                    <option value="">Select Status</option>
                                    <option value="active" {{ old('status', $customer->status) == 'active' ? 'selected' : '' }}>Active</option>
                                    <option value="inactive" {{ old('status', $customer->status) == 'inactive' ? 'selected' : '' }}>Inactive</option>
                                    <option value="blacklisted" {{ old('status', $customer->status) == 'blacklisted' ? 'selected' : '' }}>Blacklisted</option>
                                </select>
                                @error('status')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="address" class="form-label">Complete Address *</label>
                        <textarea class="form-control @error('address') is-invalid @enderror"
                                  id="address" name="address" rows="3"
                                  placeholder="Enter complete address" required>{{ old('address', $customer->address) }}</textarea>
                        @error('address')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>

                    <!-- Document Upload Section -->
                    <div class="card mt-4">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0">Document Uploads (Update if needed)</h5>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <!-- CNIC Front -->
                                <div class="col-md-3">
                                    <div class="mb-3">
                                        <label for="cnic_front" class="form-label">CNIC Front Side</label>
                                        <div class="input-group">
                                            <input type="file" class="form-control @error('cnic_front') is-invalid @enderror"
                                                   id="cnic_front" name="cnic_front">
                                        </div>
                                        <small class="form-text text-muted">Max: 2MB (JPG, PNG, GIF)</small>
                                        @error('cnic_front')
                                            <div class="invalid-feedback d-block">{{ $message }}</div>
                                        @enderror
                                        @if($customer->cnic_front)
                                            <div class="mt-2">
                                                <img src="{{ asset('storage/' . $customer->cnic_front) }}"
                                                     alt="CNIC Front" class="img-thumbnail" width="120">
                                                <div class="mt-1">
                                                    <a href="{{ asset('storage/' . $customer->cnic_front) }}"
                                                       target="_blank" class="btn btn-sm btn-info">
                                                        <i class="fas fa-eye"></i> View
                                                    </a>
                                                </div>
                                            </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- CNIC Back -->
                                <div class="col-md-3">
                                    <div class="mb-3">
                                        <label for="cnic_back" class="form-label">CNIC Back Side</label>
                                        <div class="input-group">
                                            <input type="file" class="form-control @error('cnic_back') is-invalid @enderror"
                                                   id="cnic_back" name="cnic_back">
                                        </div>
                                        <small class="form-text text-muted">Max: 2MB (JPG, PNG, GIF)</small>
                                        @error('cnic_back')
                                            <div class="invalid-feedback d-block">{{ $message }}</div>
                                        @enderror
                                        @if($customer->cnic_back)
                                            <div class="mt-2">
                                                <img src="{{ asset('storage/' . $customer->cnic_back) }}"
                                                     alt="CNIC Back" class="img-thumbnail" width="120">
                                                <div class="mt-1">
                                                    <a href="{{ asset('storage/' . $customer->cnic_back) }}"
                                                       target="_blank" class="btn btn-sm btn-info">
                                                        <i class="fas fa-eye"></i> View
                                                    </a>
                                                </div>
                                            </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Customer Picture -->
                                <div class="col-md-3">
                                    <div class="mb-3">
                                        <label for="customer_picture" class="form-label">Customer Picture</label>
                                        <div class="input-group">
                                            <input type="file" class="form-control @error('customer_picture') is-invalid @enderror"
                                                   id="customer_picture" name="customer_picture">
                                        </div>
                                        <small class="form-text text-muted">Max: 2MB (JPG, PNG, GIF)</small>
                                        @error('customer_picture')
                                            <div class="invalid-feedback d-block">{{ $message }}</div>
                                        @enderror
                                        @if($customer->customer_picture)
                                            <div class="mt-2">
                                                <img src="{{ asset('storage/' . $customer->customer_picture) }}"
                                                     alt="Customer Picture" class="img-thumbnail rounded-circle" width="120" height="120" style="object-fit: cover;">
                                                <div class="mt-1">
                                                    <a href="{{ asset('storage/' . $customer->customer_picture) }}"
                                                       target="_blank" class="btn btn-sm btn-info">
                                                        <i class="fas fa-eye"></i> View
                                                    </a>
                                                </div>
                                            </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Other Documents -->
                                <div class="col-md-3">
                                    <div class="mb-3">
                                        <label for="other_documents" class="form-label">Other Documents</label>
                                        <div class="input-group">
                                            <input type="file" class="form-control @error('other_documents.*') is-invalid @enderror"
                                                   id="other_documents" name="other_documents[]" multiple>
                                        </div>
                                        <small class="form-text text-muted">Max: 5MB each (PDF, DOC, JPG, PNG)</small>
                                        @error('other_documents.*')
                                            <div class="invalid-feedback d-block">{{ $message }}</div>
                                        @enderror
                                        @if(!empty($customer->other_documents))
                                            <div class="mt-2">
                                                @foreach($customer->other_documents as $index => $document)
                                                    <div class="d-flex justify-content-between align-items-center mb-1">
                                                        <a href="{{ asset('storage/' . $document) }}" target="_blank"
                                                           class="btn btn-sm btn-outline-info text-truncate" style="max-width: 150px;">
                                                            Document {{ $index + 1 }}
                                                        </a>
                                                    </div>
                                                @endforeach
                                            </div>
                                        @endif
                                    </div>
                                </div>
                            </div>
                            <div class="alert alert-info mt-2">
                                <i class="fas fa-info-circle me-2"></i>
                                Upload new files to replace existing ones. Leave empty to keep current documents.
                            </div>
                        </div>
                    </div>

                    <div class="mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i> Update Customer
                        </button>
                        <a href="{{ route('customers.show', $customer->id) }}" class="btn btn-info">
                            <i class="fas fa-eye me-2"></i> View Profile
                        </a>
                        <a href="{{ route('customers.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times me-2"></i> Cancel
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
    $(document).ready(function() {
        // Format phone number
        $('#phone').on('input', function() {
            let value = $(this).val().replace(/\D/g, '');
            if (value.length > 10) value = value.substring(0, 10);
            $(this).val(value);
        });

        // Format CNIC
        $('#cnic').on('input', function() {
            let value = $(this).val().replace(/\D/g, '');
            if (value.length > 13) value = value.substring(0, 13);

            // Add dashes
            if (value.length > 5) {
                value = value.substring(0, 5) + '-' + value.substring(5);
            }
            if (value.length > 13) {
                value = value.substring(0, 13) + '-' + value.substring(13);
            }
            $(this).val(value);
        });

        // Form validation
        $('#customerForm').validate({
            rules: {
                name: 'required',
                phone: {
                    required: true,
                    minlength: 10,
                    maxlength: 10,
                    digits: true
                },
                cnic: {
                    required: true,
                    minlength: 15,
                    maxlength: 15
                },
                address: 'required',
                status: 'required',
                email: {
                    email: true
                }
            },
            messages: {
                name: 'Please enter customer name',
                phone: {
                    required: 'Please enter phone number',
                    minlength: 'Phone number must be 10 digits',
                    maxlength: 'Phone number must be 10 digits',
                    digits: 'Please enter numbers only'
                },
                cnic: {
                    required: 'Please enter CNIC number',
                    minlength: 'CNIC must be 13 digits',
                    maxlength: 'CNIC must be 13 digits'
                },
                address: 'Please enter address',
                status: 'Please select status',
                email: 'Please enter a valid email address'
            },
            errorElement: 'div',
            errorClass: 'invalid-feedback',
            highlight: function(element) {
                $(element).addClass('is-invalid');
            },
            unhighlight: function(element) {
                $(element).removeClass('is-invalid');
            }
        });
    });
</script>
@endpush
@endsection
