@extends('layouts.app')

@section('title', 'Installment Details')
@section('page-title', 'Installment Details')

@section('content')
<div class="row">
    <!-- Installment Details -->
    <div class="col-md-4">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Installment Information</h5>
                <div class="dropdown">
                    <button class="btn btn-sm btn-outline-secondary dropdown-toggle" type="button"
                            data-bs-toggle="dropdown">
                        <i class="fas fa-cog"></i>
                    </button>
                    <ul class="dropdown-menu">
                        <li>
                            <a class="dropdown-item" href="{{ route('installments.edit', $installment) }}">
                                <i class="fas fa-edit me-2"></i> Edit
                            </a>
                        </li>
                        <li>
                            <a class="dropdown-item" href="{{ route('payments.create', ['installment_id' => $installment->id]) }}">
                                <i class="fas fa-credit-card me-2"></i> Record Payment
                            </a>
                        </li>
                        @if($installment->status != 'paid')
                        <li>
                            <form action="{{ route('installments.mark-paid', $installment) }}" method="POST"
                                  class="dropdown-item p-0">
                                @csrf
                                <button type="submit" class="btn btn-link text-success p-0"
                                        onclick="return confirm('Mark this installment as paid?')">
                                    <i class="fas fa-check me-2"></i> Mark as Paid
                                </button>
                            </form>
                        </li>
                        @endif
                        <li><hr class="dropdown-divider"></li>
                        <li>
                            <form action="{{ route('installments.destroy', $installment) }}" method="POST"
                                  class="dropdown-item p-0">
                                @csrf
                                @method('DELETE')
                                <button type="submit" class="btn btn-link text-danger p-0"
                                        onclick="return confirm('Are you sure? This will delete all associated payments.')">
                                    <i class="fas fa-trash me-2"></i> Delete
                                </button>
                            </form>
                        </li>
                    </ul>
                </div>
            </div>
            <div class="card-body">
                <div class="text-center mb-4">
                    <div class="avatar bg-primary text-white rounded-circle d-flex align-items-center justify-content-center mx-auto mb-3"
                         style="width: 80px; height: 80px; font-size: 32px;">
                        {{ $installment->installment_number }}
                    </div>
                    <h4 class="mt-2">Installment #{{ $installment->installment_number }}</h4>
                    <p class="text-muted mb-3">
                        @php
                            $statusColors = [
                                'paid' => 'success',
                                'pending' => 'warning',
                                'overdue' => 'danger',
                                'partial' => 'info'
                            ];
                            $statusIcons = [
                                'paid' => 'check-circle',
                                'pending' => 'clock',
                                'overdue' => 'exclamation-triangle',
                                'partial' => 'money-check-alt'
                            ];
                        @endphp
                        <span class="badge bg-{{ $statusColors[$installment->status] }} p-2 fs-6">
                            <i class="fas fa-{{ $statusIcons[$installment->status] }} me-1"></i>
                            {{ ucfirst($installment->status) }}
                        </span>
                    </p>

                    @if($installment->isOverdue())
                        <div class="alert alert-danger py-2">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong>Overdue!</strong> {{ now()->diffInDays($installment->due_date) }} days overdue
                        </div>
                    @endif
                </div>

                <!-- Property Information -->
                <div class="mb-4">
                    <h6 class="border-bottom pb-2">
                        <i class="fas fa-building me-2 text-primary"></i>Property Information
                    </h6>
                    <div class="d-flex align-items-center mb-3">
                        <div class="flex-shrink-0">
                            <div class="avatar bg-primary text-white rounded-circle d-flex align-items-center justify-content-center"
                                 style="width: 50px; height: 50px;">
                                <i class="fas fa-home"></i>
                            </div>
                        </div>
                        <div class="flex-grow-1 ms-3">
                            <h6 class="mb-0">{{ $installment->property->name }}</h6>
                            <p class="small text-muted mb-1">
                                <i class="fas fa-map-marker-alt"></i> {{ $installment->property->location }}
                            </p>
                            <p class="small text-muted mb-0">
                                Total: <strong>Rs. {{ number_format($installment->property->total_price, 2) }}</strong>
                            </p>
                        </div>
                        <div class="flex-shrink-0">
                            <a href="{{ route('properties.show', $installment->property) }}"
                               class="btn btn-sm btn-outline-primary">
                                <i class="fas fa-external-link-alt"></i>
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Customer Information -->
                <div class="mb-4">
                    <h6 class="border-bottom pb-2">
                        <i class="fas fa-user me-2 text-info"></i>Customer Information
                    </h6>
                    <div class="d-flex align-items-center mb-3">
                        <div class="flex-shrink-0">
                            <div class="avatar bg-info text-white rounded-circle d-flex align-items-center justify-content-center"
                                 style="width: 50px; height: 50px; font-size: 18px;">
                                {{ substr($installment->customer->name, 0, 1) }}
                            </div>
                        </div>
                        <div class="flex-grow-1 ms-3">
                            <h6 class="mb-0">{{ $installment->customer->name }}</h6>
                            <p class="small text-muted mb-1">
                                <i class="fas fa-phone"></i> {{ $installment->customer->phone }}
                            </p>
                            <p class="small text-muted mb-0">
                                <i class="fas fa-envelope"></i> {{ $installment->customer->email }}
                            </p>
                        </div>
                        <div class="flex-shrink-0">
                            <a href="{{ route('customers.show', $installment->customer) }}"
                               class="btn btn-sm btn-outline-info">
                                <i class="fas fa-external-link-alt"></i>
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Financial Details -->
                <div class="mb-4">
                    <h6 class="border-bottom pb-2">
                        <i class="fas fa-money-bill-wave me-2 text-success"></i>Financial Details
                    </h6>
                    <div class="row">
                        <div class="col-6 mb-3">
                            <small class="text-muted">Installment Amount</small>
                            <p class="mb-1 fs-5 fw-bold text-primary">
                                Rs. {{ number_format($installment->amount, 2) }}
                            </p>
                        </div>
                        <div class="col-6 mb-3">
                            <small class="text-muted">Paid Amount</small>
                            <p class="mb-1 fs-5 fw-bold text-success">
                                Rs. {{ number_format($installment->paid_amount, 2) }}
                            </p>
                        </div>
                        <div class="col-6">
                            <small class="text-muted">Remaining Amount</small>
                            <p class="mb-1 fs-5 fw-bold text-warning">
                                Rs. {{ number_format($installment->remaining_amount, 2) }}
                            </p>
                        </div>
                        <div class="col-6">
                            <small class="text-muted">Due Date</small>
                            <p class="mb-1">
                                <strong>{{ $installment->due_date->format('d M Y') }}</strong>
                            </p>
                            <p class="small text-muted mb-0">
                                ({{ $installment->due_date->diffForHumans() }})
                            </p>
                        </div>
                    </div>
                </div>

                <!-- Payment Progress -->
                <div class="mb-4">
                    <div class="d-flex justify-content-between mb-2">
                        <small class="text-muted">Payment Progress</small>
                        <small class="text-muted">
                            {{ round(($installment->paid_amount / $installment->amount) * 100) }}%
                        </small>
                    </div>
                    <div class="progress" style="height: 10px;">
                        <div class="progress-bar bg-success" role="progressbar"
                             style="width: {{ ($installment->paid_amount / $installment->amount) * 100 }}%;"
                             aria-valuenow="{{ ($installment->paid_amount / $installment->amount) * 100 }}"
                             aria-valuemin="0" aria-valuemax="100">
                        </div>
                    </div>
                    <div class="d-flex justify-content-between mt-1">
                        <small class="text-muted">Rs. 0</small>
                        <small class="text-muted">Rs. {{ number_format($installment->amount, 0) }}</small>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="mt-4">
                    <div class="d-grid gap-2">
                        @if($installment->status != 'paid')
                            <a href="{{ route('payments.create', ['installment_id' => $installment->id]) }}"
                               class="btn btn-success">
                                <i class="fas fa-credit-card me-2"></i> Record Payment
                            </a>
                        @endif
                        <a href="{{ route('installments.edit', $installment) }}"
                           class="btn btn-warning">
                            <i class="fas fa-edit me-2"></i> Edit Installment
                        </a>
                        <button type="button" class="btn btn-outline-primary" onclick="window.print()">
                            <i class="fas fa-print me-2"></i> Print Details
                        </button>
                    </div>
                </div>

                @if($installment->notes)
                <div class="mt-4">
                    <h6 class="border-bottom pb-2">
                        <i class="fas fa-sticky-note me-2 text-secondary"></i>Notes
                    </h6>
                    <div class="alert alert-light">
                        <p class="mb-0">{{ $installment->notes }}</p>
                    </div>
                </div>
                @endif
            </div>
        </div>
    </div>

    <!-- Payment History & Details -->
    <div class="col-md-8">
        <!-- Payment History -->
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Payment History</h5>
                <div>
                    <span class="badge bg-primary">
                        {{ $installment->payments->count() }} payment(s)
                    </span>
                    <a href="{{ route('payments.create', ['installment_id' => $installment->id]) }}"
                       class="btn btn-primary btn-sm ms-2">
                        <i class="fas fa-plus me-1"></i> Add Payment
                    </a>
                </div>
            </div>
            <div class="card-body">
                @if($installment->payments->count() > 0)
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Transaction ID</th>
                                <th>Date</th>
                                <th>Amount</th>
                                <th>Method</th>
                                <th>Reference</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($installment->payments as $payment)
                            <tr>
                                <td>{{ $loop->iteration }}</td>
                                <td>
                                    <code>{{ $payment->transaction_id }}</code>
                                    @if($payment->remarks)
                                    <div class="small text-muted">{{ Str::limit($payment->remarks, 30) }}</div>
                                    @endif
                                </td>
                                <td>
                                    {{ $payment->payment_date->format('d M Y') }}
                                    <div class="small text-muted">{{ $payment->payment_date->format('h:i A') }}</div>
                                </td>
                                <td>
                                    <span class="fw-bold">Rs. {{ number_format($payment->amount, 2) }}</span>
                                </td>
                                <td>
                                    @php
                                        $methodColors = [
                                            'cash' => 'secondary',
                                            'bank_transfer' => 'primary',
                                            'cheque' => 'info',
                                            'online' => 'success'
                                        ];
                                    @endphp
                                    <span class="badge bg-{{ $methodColors[$payment->payment_method] ?? 'secondary' }}">
                                        {{ ucfirst(str_replace('_', ' ', $payment->payment_method)) }}
                                    </span>
                                </td>
                                <td>
                                    @if($payment->reference_number)
                                        <code>{{ $payment->reference_number }}</code>
                                    @else
                                        <span class="text-muted">N/A</span>
                                    @endif
                                </td>
                                <td>
                                    <span class="badge bg-{{ $payment->status == 'completed' ? 'success' :
                                                              ($payment->status == 'pending' ? 'warning' :
                                                              ($payment->status == 'failed' ? 'danger' : 'secondary')) }}">
                                        {{ ucfirst($payment->status) }}
                                    </span>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm" role="group">
                                        <a href="{{ route('payments.show', $payment) }}"
                                           class="btn btn-info" title="View Details">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <a href="{{ route('payments.edit', $payment) }}"
                                           class="btn btn-warning" title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <form action="{{ route('payments.destroy', $payment) }}"
                                              method="POST" class="d-inline">
                                            @csrf
                                            @method('DELETE')
                                            <button type="submit" class="btn btn-danger"
                                                    onclick="return confirm('Delete this payment?')"
                                                    title="Delete">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                        <tfoot>
                            <tr class="table-success">
                                <td colspan="3" class="text-end"><strong>Total Paid:</strong></td>
                                <td colspan="5">
                                    <strong>Rs. {{ number_format($installment->payments->where('status', 'completed')->sum('amount'), 2) }}</strong>
                                </td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
                @else
                <div class="text-center py-5">
                    <div class="empty-state">
                        <i class="fas fa-credit-card fa-4x text-muted mb-4"></i>
                        <h4>No Payments Recorded</h4>
                        <p class="text-muted">No payments have been recorded for this installment yet.</p>
                        <a href="{{ route('payments.create', ['installment_id' => $installment->id]) }}"
                           class="btn btn-primary">
                            <i class="fas fa-plus me-2"></i> Record First Payment
                        </a>
                    </div>
                </div>
                @endif
            </div>
        </div>

        <!-- Installment Timeline -->
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="mb-0">Installment Timeline</h5>
            </div>
            <div class="card-body">
                <div class="timeline">
                    <!-- Timeline Item 1: Creation -->
                    <div class="timeline-item">
                        <div class="timeline-marker bg-primary">
                            <i class="fas fa-calendar-plus text-white"></i>
                        </div>
                        <div class="timeline-content">
                            <div class="timeline-title">Installment Created</div>
                            <div class="timeline-date">{{ $installment->created_at->format('d M Y, h:i A') }}</div>
                            <div class="timeline-description">Installment schedule was created</div>
                        </div>
                    </div>

                    <!-- Timeline Item 2: Due Date -->
                    <div class="timeline-item">
                        <div class="timeline-marker bg-{{ $installment->isOverdue() ? 'danger' : 'info' }}">
                            <i class="fas fa-calendar-check text-white"></i>
                        </div>
                        <div class="timeline-content">
                            <div class="timeline-title">Due Date</div>
                            <div class="timeline-date">{{ $installment->due_date->format('d M Y') }}</div>
                            <div class="timeline-description">
                                Payment due date
                                @if($installment->isOverdue())
                                    <span class="badge bg-danger ms-2">Overdue</span>
                                @endif
                            </div>
                        </div>
                    </div>

                    <!-- Timeline Item 3: First Payment -->
                    @if($installment->paid_amount > 0)
                    <div class="timeline-item">
                        <div class="timeline-marker bg-success">
                            <i class="fas fa-money-check-alt text-white"></i>
                        </div>
                        <div class="timeline-content">
                            <div class="timeline-title">First Payment</div>
                            <div class="timeline-date">
                                {{ $installment->payments->first()->payment_date->format('d M Y') }}
                            </div>
                            <div class="timeline-description">
                                Rs. {{ number_format($installment->payments->first()->amount, 2) }} payment recorded
                            </div>
                        </div>
                    </div>
                    @endif

                    <!-- Timeline Item 4: Last Update -->
                    <div class="timeline-item">
                        <div class="timeline-marker bg-secondary">
                            <i class="fas fa-sync-alt text-white"></i>
                        </div>
                        <div class="timeline-content">
                            <div class="timeline-title">Last Updated</div>
                            <div class="timeline-date">{{ $installment->updated_at->format('d M Y, h:i A') }}</div>
                            <div class="timeline-description">
                                Status: <span class="badge bg-{{ $statusColors[$installment->status] }}">
                                    {{ ucfirst($installment->status) }}
                                </span>
                            </div>
                        </div>
                    </div>

                    <!-- Timeline Item 5: Fully Paid (if applicable) -->
                    @if($installment->status == 'paid')
                    <div class="timeline-item">
                        <div class="timeline-marker bg-success">
                            <i class="fas fa-check-circle text-white"></i>
                        </div>
                        <div class="timeline-content">
                            <div class="timeline-title">Fully Paid</div>
                            <div class="timeline-date">
                                {{ $installment->payments->max('payment_date')->format('d M Y') }}
                            </div>
                            <div class="timeline-description">Installment completed successfully</div>
                        </div>
                    </div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Related Installments -->
        <div class="card mt-4">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Related Installments</h5>
                <a href="{{ route('properties.show', $installment->property) }}" class="btn btn-sm btn-outline-primary">
                    View All
                </a>
            </div>
            <div class="card-body">
                @php
                    $relatedInstallments = $installment->property->installments()
                        ->where('id', '!=', $installment->id)
                        ->with('customer')
                        ->orderBy('installment_number', 'asc')
                        ->take(5)
                        ->get();
                @endphp

                @if($relatedInstallments->count() > 0)
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Customer</th>
                                <th>Due Date</th>
                                <th>Amount</th>
                                <th>Paid</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($relatedInstallments as $related)
                            <tr class="{{ $related->id == $installment->id ? 'table-active' : '' }}">
                                <td>
                                    <strong>#{{ $related->installment_number }}</strong>
                                </td>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="avatar bg-info text-white rounded-circle d-flex align-items-center justify-content-center me-2"
                                             style="width: 25px; height: 25px; font-size: 10px;">
                                            {{ substr($related->customer->name, 0, 1) }}
                                        </div>
                                        <span>{{ Str::limit($related->customer->name, 15) }}</span>
                                    </div>
                                </td>
                                <td>
                                    {{ $related->due_date->format('d/m/Y') }}
                                    @if($related->isOverdue())
                                        <br><small class="text-danger">Overdue</small>
                                    @endif
                                </td>
                                <td>Rs. {{ number_format($related->amount, 2) }}</td>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="progress flex-grow-1 me-2" style="height: 5px;">
                                            <div class="progress-bar bg-success"
                                                 style="width: {{ ($related->paid_amount / $related->amount) * 100 }}%">
                                            </div>
                                        </div>
                                        <small>{{ round(($related->paid_amount / $related->amount) * 100) }}%</small>
                                    </div>
                                </td>
                                <td>
                                    <span class="badge bg-{{ $statusColors[$related->status] }}">
                                        {{ ucfirst($related->status) }}
                                    </span>
                                </td>
                                <td>
                                    <a href="{{ route('installments.show', $related) }}"
                                       class="btn btn-sm btn-outline-info">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    @if($related->status != 'paid')
                                    <a href="{{ route('payments.create', ['installment_id' => $related->id]) }}"
                                       class="btn btn-sm btn-outline-success">
                                        <i class="fas fa-credit-card"></i>
                                    </a>
                                    @endif
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                @else
                <div class="text-center py-3">
                    <p class="text-muted mb-0">No other installments for this property.</p>
                </div>
                @endif
            </div>
        </div>

        <!-- Quick Stats -->
        <div class="row mt-4">
            <div class="col-md-4">
                <div class="card bg-primary text-white">
                    <div class="card-body text-center py-3">
                        <h2 class="mb-0">{{ $installment->installment_number }}</h2>
                        <small>Installment Number</small>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card bg-success text-white">
                    <div class="card-body text-center py-3">
                        <h2 class="mb-0">{{ $installment->payments->count() }}</h2>
                        <small>Total Payments</small>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card bg-info text-white">
                    <div class="card-body text-center py-3">
                        <h2 class="mb-0">{{ round(($installment->paid_amount / $installment->amount) * 100) }}%</h2>
                        <small>Completion</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Print View (Hidden) -->
<div id="printView" class="d-none">
    <div class="container p-5">
        <div class="text-center mb-4">
            <h2>Installment Details</h2>
            <h4>Installment #{{ $installment->installment_number }}</h4>
            <p>Generated on: {{ now()->format('d M Y, h:i A') }}</p>
        </div>

        <div class="row">
            <div class="col-md-6">
                <h5>Property Information</h5>
                <p><strong>Name:</strong> {{ $installment->property->name }}</p>
                <p><strong>Location:</strong> {{ $installment->property->location }}</p>
                <p><strong>Total Price:</strong> Rs. {{ number_format($installment->property->total_price, 2) }}</p>
            </div>
            <div class="col-md-6">
                <h5>Customer Information</h5>
                <p><strong>Name:</strong> {{ $installment->customer->name }}</p>
                <p><strong>Phone:</strong> {{ $installment->customer->phone }}</p>
                <p><strong>Email:</strong> {{ $installment->customer->email }}</p>
            </div>
        </div>

        <hr>

        <div class="row">
            <div class="col-md-6">
                <h5>Installment Details</h5>
                <p><strong>Amount:</strong> Rs. {{ number_format($installment->amount, 2) }}</p>
                <p><strong>Paid Amount:</strong> Rs. {{ number_format($installment->paid_amount, 2) }}</p>
                <p><strong>Remaining Amount:</strong> Rs. {{ number_format($installment->remaining_amount, 2) }}</p>
            </div>
            <div class="col-md-6">
                <h5>Dates</h5>
                <p><strong>Due Date:</strong> {{ $installment->due_date->format('d M Y') }}</p>
                <p><strong>Status:</strong> {{ ucfirst($installment->status) }}</p>
                <p><strong>Created:</strong> {{ $installment->created_at->format('d M Y') }}</p>
            </div>
        </div>

        @if($installment->payments->count() > 0)
        <hr>
        <h5>Payment History</h5>
        <table class="table table-bordered">
            <thead>
                <tr>
                    <th>Transaction ID</th>
                    <th>Date</th>
                    <th>Amount</th>
                    <th>Method</th>
                    <th>Status</th>
                </tr>
            </thead>
            <tbody>
                @foreach($installment->payments as $payment)
                <tr>
                    <td>{{ $payment->transaction_id }}</td>
                    <td>{{ $payment->payment_date->format('d M Y') }}</td>
                    <td>Rs. {{ number_format($payment->amount, 2) }}</td>
                    <td>{{ ucfirst(str_replace('_', ' ', $payment->payment_method)) }}</td>
                    <td>{{ ucfirst($payment->status) }}</td>
                </tr>
                @endforeach
                <tr>
                    <td colspan="2" class="text-end"><strong>Total:</strong></td>
                    <td colspan="3">
                        <strong>Rs. {{ number_format($installment->payments->where('status', 'completed')->sum('amount'), 2) }}</strong>
                    </td>
                </tr>
            </tbody>
        </table>
        @endif

        <hr>
        <div class="text-center mt-4">
            <p>This is a computer generated document. No signature required.</p>
        </div>
    </div>
</div>

@push('scripts')
<script>
    // Print functionality
    function printDetails() {
        const printContent = document.getElementById('printView').innerHTML;
        const originalContent = document.body.innerHTML;

        document.body.innerHTML = printContent;
        window.print();
        document.body.innerHTML = originalContent;
        window.location.reload();
    }

    // Initialize tooltips
    $(document).ready(function() {
        $('[data-bs-toggle="tooltip"]').tooltip();

        // Auto-refresh status for overdue installments
        @if($installment->status == 'pending' && $installment->isOverdue())
        setInterval(function() {
            fetch("{{ route('installments.show', $installment) }}", {
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.text())
            .then(html => {
                // You could update the status badge here if needed
                console.log('Status checked for overdue installment');
            });
        }, 60000); // Check every minute
        @endif
    });
</script>
@endpush

@push('styles')
<style>
    /* Avatar Styles */
    .avatar {
        font-weight: bold;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    /* Badge Styles */
    .badge {
        min-width: 80px;
        text-align: center;
        padding: 8px 12px;
    }

    /* Progress Bar Styles */
    .progress {
        border-radius: 10px;
        overflow: hidden;
    }

    /* Timeline Styles */
    .timeline {
        position: relative;
        padding-left: 40px;
    }

    .timeline::before {
        content: '';
        position: absolute;
        left: 20px;
        top: 0;
        bottom: 0;
        width: 2px;
        background: linear-gradient(to bottom, #dee2e6, #6c757d);
    }

    .timeline-item {
        position: relative;
        margin-bottom: 25px;
    }

    .timeline-marker {
        position: absolute;
        left: -40px;
        top: 0;
        width: 40px;
        height: 40px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        border: 3px solid white;
        box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }

    .timeline-content {
        padding-left: 20px;
    }

    .timeline-title {
        font-weight: 600;
        color: #495057;
        margin-bottom: 5px;
    }

    .timeline-date {
        font-size: 12px;
        color: #6c757d;
        margin-bottom: 5px;
    }

    .timeline-description {
        font-size: 14px;
        color: #495057;
    }

    /* Empty State */
    .empty-state {
        opacity: 0.7;
    }

    /* Hover Effects */
    .table-hover tbody tr:hover {
        background-color: rgba(67, 97, 238, 0.05);
    }

    /* Card Header Styles */
    .card-header {
        background-color: rgba(67, 97, 238, 0.05);
        border-bottom: 1px solid rgba(67, 97, 238, 0.1);
    }

    /* Print Styles */
    @media print {
        .no-print {
            display: none !important;
        }

        body {
            background: white !important;
            color: black !important;
        }

        .card {
            border: none !important;
            box-shadow: none !important;
        }
    }
</style>
@endpush
@endsection
