@extends('layouts.app')

@section('title', 'Add New Property')
@section('page-title', 'Add New Property')

@section('content')
<div class="row justify-content-center">
    <div class="col-md-10">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Property Information</h5>
            </div>
            <div class="card-body">
                <form method="POST" action="{{ route('properties.store') }}" id="propertyForm">
                    @csrf

                    <div class="row">
                        <!-- Property Details -->
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="name" class="form-label">Property Name *</label>
                                <input type="text" class="form-control @error('name') is-invalid @enderror"
                                       id="name" name="name" value="{{ old('name') }}"
                                       placeholder="Enter property name" required>
                                @error('name')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="location" class="form-label">Location *</label>
                                <textarea class="form-control @error('location') is-invalid @enderror"
                                          id="location" name="location" rows="2"
                                          placeholder="Enter property address" required>{{ old('location') }}</textarea>
                                @error('location')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="total_price" class="form-label">Total Price (PKR) *</label>
                                <div class="input-group">
                                    <span class="input-group-text">Rs.</span>
                                    <input type="number" step="0.01" class="form-control @error('total_price') is-invalid @enderror"
                                           id="total_price" name="total_price" value="{{ old('total_price') }}"
                                           placeholder="Enter total price" required min="0">
                                </div>
                                @error('total_price')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="down_payment" class="form-label">Down Payment (PKR) *</label>
                                <div class="input-group">
                                    <span class="input-group-text">Rs.</span>
                                    <input type="number" step="0.01" class="form-control @error('down_payment') is-invalid @enderror"
                                           id="down_payment" name="down_payment" value="{{ old('down_payment') }}"
                                           placeholder="Enter down payment" required min="0">
                                </div>
                                @error('down_payment')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <!-- Installment Details -->
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="installment_years" class="form-label">Installment Period (Years) *</label>
                                <select class="form-control @error('installment_years') is-invalid @enderror"
                                        id="installment_years" name="installment_years" required>
                                    <option value="">Select years</option>
                                    @for($i = 1; $i <= 30; $i++)
                                        <option value="{{ $i }}" {{ old('installment_years') == $i ? 'selected' : '' }}>
                                            {{ $i }} {{ Str::plural('year', $i) }}
                                        </option>
                                    @endfor
                                </select>
                                @error('installment_years')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="mb-3">
                                <label for="customer_id" class="form-label">Select Customer *</label>
                                <select class="form-control @error('customer_id') is-invalid @enderror"
                                        id="customer_id" name="customer_id" required>
                                    <option value="">Select a customer</option>
                                    @foreach($customers as $customer)
                                        <option value="{{ $customer->id }}" {{ old('customer_id') == $customer->id ? 'selected' : '' }}>
                                            {{ $customer->name }} - {{ $customer->phone }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('customer_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="mt-2">
                                    <a href="{{ route('customers.create') }}" class="btn btn-sm btn-outline-primary">
                                        <i class="fas fa-plus"></i> Add New Customer
                                    </a>
                                </div>
                            </div>

                            <div class="mb-3">
                                <label for="description" class="form-label">Description</label>
                                <textarea class="form-control @error('description') is-invalid @enderror"
                                          id="description" name="description" rows="3"
                                          placeholder="Enter property description">{{ old('description') }}</textarea>
                                @error('description')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Calculation Preview -->
                            <div class="card bg-light">
                                <div class="card-body">
                                    <h6 class="card-title">Installment Preview</h6>
                                    <div class="row">
                                        <div class="col-6">
                                            <small class="text-muted">Installment Amount:</small>
                                            <p class="mb-1"><strong id="preview_installment_amount">Rs. 0.00</strong></p>
                                        </div>
                                        <div class="col-6">
                                            <small class="text-muted">Total Installments:</small>
                                            <p class="mb-1"><strong id="preview_total_installments">0</strong></p>
                                        </div>
                                        <div class="col-12 mt-2">
                                            <small class="text-muted">Payment After Down:</small>
                                            <p class="mb-0"><strong id="preview_remaining_amount">Rs. 0.00</strong></p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="row mt-4">
                        <div class="col-md-12">
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                <strong>Note:</strong> After creating the property, the system will automatically generate monthly installments for the selected customer.
                            </div>
                        </div>
                    </div>

                    <div class="mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i> Create Property & Generate Installments
                        </button>
                        <a href="{{ route('properties.index') }}" class="btn btn-secondary">
                            <i class="fas fa-times me-2"></i> Cancel
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
    $(document).ready(function() {
        // Calculate installment preview
        function calculatePreview() {
            const totalPrice = parseFloat($('#total_price').val()) || 0;
            const downPayment = parseFloat($('#down_payment').val()) || 0;
            const years = parseInt($('#installment_years').val()) || 0;

            if (years > 0 && totalPrice > 0) {
                const remainingAmount = totalPrice - downPayment;
                const monthlyInstallment = remainingAmount / (years * 12);
                const totalInstallments = years * 12;

                $('#preview_installment_amount').text('Rs. ' + monthlyInstallment.toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,'));
                $('#preview_total_installments').text(totalInstallments);
                $('#preview_remaining_amount').text('Rs. ' + remainingAmount.toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,'));
            }
        }

        // Trigger calculation on input change
        $('#total_price, #down_payment, #installment_years').on('input change', calculatePreview);

        // Form validation
        $('#propertyForm').validate({
            rules: {
                name: 'required',
                location: 'required',
                total_price: {
                    required: true,
                    min: 100000
                },
                down_payment: {
                    required: true,
                    min: 0
                },
                installment_years: {
                    required: true,
                    min: 1
                },
                customer_id: 'required'
            },
            messages: {
                name: 'Please enter property name',
                location: 'Please enter property location',
                total_price: {
                    required: 'Please enter total price',
                    min: 'Minimum price should be Rs. 100,000'
                },
                down_payment: {
                    required: 'Please enter down payment',
                    min: 'Down payment cannot be negative'
                },
                installment_years: {
                    required: 'Please select installment period',
                    min: 'Minimum period is 1 year'
                },
                customer_id: 'Please select a customer'
            },
            errorElement: 'div',
            errorClass: 'invalid-feedback',
            highlight: function(element) {
                $(element).addClass('is-invalid');
            },
            unhighlight: function(element) {
                $(element).removeClass('is-invalid');
            }
        });

        // Initial calculation
        calculatePreview();
    });
</script>
@endpush
@endsection
